package com.unad.test;


import static androidx.recyclerview.widget.RecyclerView.SCROLL_STATE_IDLE;

import android.app.Activity;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.TextView;
import android.widget.Toast;

import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.unad.samples.R;
import com.unad.sdk.UNADFeedAd;
import com.unad.sdk.UNADFeedAdView;
import com.unad.sdk.dto.AdError;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * 在消息流中接入原生模板广告的示例
 */

public class FeedRecyclerViewActivity extends Activity {
    private static final String TAG = "UNAD_SDK";
    public static final int ITEMS_COUNT = 10;
    public static int FIRST_AD_POSITION = 2; // 第一条广告的位置
    public static int ITEMS_PER_AD = 5;     // 每间隔多少个条目插入一条广告
    private boolean mIsLoading = true;
    private RecyclerView mRecyclerView;
    private LinearLayoutManager mLinearLayoutManager;
    private CustomAdapter mAdapter;
    private List<NormalItem> mNormalDataList = new ArrayList<NormalItem>();
    private UNADFeedAd mADManager;
    private List<UNADFeedAdView> mAdViewList = new ArrayList<>();
    private HashMap<UNADFeedAdView, Integer> mAdViewPositionMap = new HashMap<UNADFeedAdView, Integer>();

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        requestWindowFeature(Window.FEATURE_NO_TITLE);
        super.onCreate(savedInstanceState);
        setContentView(R.layout.feed_recycler_view);
        mRecyclerView = (RecyclerView) findViewById(R.id.recycler_view);
        mRecyclerView.setHasFixedSize(true);
        mLinearLayoutManager = new LinearLayoutManager(this);
        mRecyclerView.setLayoutManager(mLinearLayoutManager);
        mRecyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
            @Override
            public void onScrollStateChanged( RecyclerView recyclerView, int newState) {
                super.onScrollStateChanged(recyclerView, newState);


                //滚动到底再次加载广告
                if (!mIsLoading && newState == SCROLL_STATE_IDLE && !recyclerView.canScrollVertically(1)) {
                    Toast.makeText(FeedRecyclerViewActivity.this,"开始拉取广告",Toast.LENGTH_LONG).show();
                    mIsLoading = true;
                    mADManager.loadAd();
                }
            }
        });
        initData();
    }

    public  void back(View view){
        finish();
    }

    @Override
    protected void onDestroy() {
        // 使用完了每一个UNADFeedAdView之后都要释放掉资源。
        if (mAdViewList != null) {
            for (UNADFeedAdView view : mAdViewList) {
                view.destroy();
            }
        }
        if(null!=mADManager){
            mADManager.release();
        }
        super.onDestroy();


    }


    @Override
    protected void onResume() {
        super.onResume();
        if(mADManager != null){
            mADManager.onResume();
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        if(mADManager != null){
            mADManager.onPause();
        }
    }


    private void initData() {
        mAdapter = new CustomAdapter(mNormalDataList);
        mRecyclerView.setAdapter(mAdapter);
        initNativeExpressAD();

        for (int i = 0; i < ITEMS_COUNT; ++i) {
            mNormalDataList.add(new NormalItem(" 正文 : " +i));
            mAdapter.notifyItemInserted( i);
        }

    }

    private void initNativeExpressAD() {
        mADManager = new UNADFeedAd(this, "UNAD_TEST_UNIT_ID", new UNADFeedAd.AdViewListener(){

            @Override
            public void onAdLoaded(UNADFeedAdView unadFeedAdView) {
                mIsLoading = false;
                int count = mAdapter.getItemCount();
                int adCount = mAdViewList.size();
                for (int i = 0; i < ITEMS_COUNT; ++i) {
                    mNormalDataList.add(new NormalItem(" 正文 : " + (count + i - adCount)));
                    mAdapter.notifyItemInserted(count + i);
                }
                int position = count + FIRST_AD_POSITION + ITEMS_PER_AD * 0;
                if (position < mNormalDataList.size()) {
                    mAdViewPositionMap.put(unadFeedAdView, position); // 把每个广告在列表中位置记录下来
                    mAdapter.addADViewToPosition(position, unadFeedAdView);
                    mAdapter.notifyItemInserted(position);
                }
                mAdViewList.add(unadFeedAdView);
            }

            @Override
            public void onAdFailed(AdError error) {
                Log.e(TAG, "返回错误:"+error.getMessage()+"==="+error.getCode());
                mIsLoading = false;
                Toast.makeText(FeedRecyclerViewActivity.this,"拉取广告失败",Toast.LENGTH_LONG).show();


            }

            @Override
            public void onAdClicked(UNADFeedAdView unadFeedAdView) {

            }

            @Override
            public void onAdClose(UNADFeedAdView unadFeedAdView) {
                //移除广告布局
                Log.e(TAG, "onAdClose");

            }

            @Override
            public void onAdOpen(UNADFeedAdView unadFeedAdView) {

            }


        });


        mADManager.loadAd();
    }


    public class NormalItem {
        private String title;

        public NormalItem(String title) {
            this.title = title;
        }

        public String getTitle() {
            return title;
        }

        public void setTitle(String title) {
            this.title = title;
        }
    }

    /**
     * RecyclerView的Adapter
     */
    class CustomAdapter extends RecyclerView.Adapter<CustomAdapter.CustomViewHolder> {

        static final int TYPE_DATA = 0;
        static final int TYPE_AD = 1;
        private List<Object> mData;

        public CustomAdapter(List list) {
            mData = list;
        }

        // 返回的UNADFeedAdView添加到数据集里面去
        public void addADViewToPosition(int position, UNADFeedAdView adView) {
            if (position >= 0 && position < mData.size() && adView != null) {
                mData.add(position, adView);
            }
        }

        // 移除UNADFeedAdView的时候是一条一条移除的
        public void removeADView(int position, UNADFeedAdView adView) {
            mData.remove(position);
            mAdapter.notifyItemRemoved(position); // position为adView在当前列表中的位置
            mAdapter.notifyItemRangeChanged(0, mData.size() - 1);
        }

        @Override
        public int getItemCount() {
            if (mData != null) {
                return mData.size();
            } else {
                return 0;
            }
        }

        @Override
        public int getItemViewType(int position) {
            return mData.get(position) instanceof UNADFeedAdView ? TYPE_AD : TYPE_DATA;
        }

        @Override
        public void onBindViewHolder(final CustomViewHolder customViewHolder, final int position) {
            int type = getItemViewType(position);
            if (TYPE_AD == type) {
                final UNADFeedAdView adView = (UNADFeedAdView) mData.get(position);
                mAdViewPositionMap.put(adView, position); // 广告在列表中的位置是可以被更新的
                if (customViewHolder.container.getChildCount() > 0
                        && customViewHolder.container.getChildAt(0) == adView) {
                    return;
                }

                if (customViewHolder.container.getChildCount() > 0) {
                    customViewHolder.container.removeAllViews();
                }

                //移除掉广告
                if (adView.getParent() != null) {
                    ((ViewGroup) adView.getParent()).removeView(adView);
                }

                customViewHolder.container.addView(adView);
                // 调用render才会开始展示广告
               adView.render();
            } else {
                customViewHolder.title.setText(((NormalItem) mData.get(position)).getTitle());
            }
        }

        @Override
        public CustomViewHolder onCreateViewHolder(ViewGroup viewGroup, int viewType) {
            int layoutId = (viewType == TYPE_AD) ? R.layout.item_ad : R.layout.item_data;
            View view = LayoutInflater.from(viewGroup.getContext()).inflate(layoutId, null);
            return new CustomViewHolder(view);
        }

        class CustomViewHolder extends RecyclerView.ViewHolder {
            public TextView title;
            public ViewGroup container;

            public CustomViewHolder(View view) {
                super(view);
                title = (TextView) view.findViewById(R.id.title);
                container = (ViewGroup) view.findViewById(R.id.express_ad_container);
            }
        }
    }

}